// umogi2.c

// PIC24FJ256GB108 Configuration Bit Settings

// CONFIG3
#pragma config WPFP = WPFP511           // Write Protection Flash Page Segment Boundary (Highest Page (same as page 170))
#pragma config WPDIS = WPDIS            // Segment Write Protection Disable bit (Segmented code protection disabled)
#pragma config WPCFG = WPCFGDIS         // Configuration Word Code Page Protection Select bit (Last page(at the top of program memory) and Flash configuration words are not protected)
#pragma config WPEND = WPENDMEM         // Segment Write Protection End Page Select bit (Write Protect from WPFP to the last page of memory)

// CONFIG2
#pragma config POSCMOD = HS             // Primary Oscillator Select (HS oscillator mode selected)
#pragma config DISUVREG = OFF            // Internal USB 3.3V Regulator Disable bit (Regulator is disabled)
#pragma config IOL1WAY = ON             // IOLOCK One-Way Set Enable bit (Write RP Registers Once)
#pragma config OSCIOFNC = OFF           // Primary Oscillator Output Function (OSCO functions as CLKO (FOSC/2))
#pragma config FCKSM = CSDCMD           // Clock Switching, Clock Monitor are disabled
#pragma config FNOSC = PRIPLL           // Oscillator Select (Primary oscillator +pll )
#pragma config PLL_96MHZ = ON           // 96MHz PLL Disable (Enabled)
#pragma config PLLDIV = DIV3           // USB 96 MHz PLL Prescaler Select bits (Oscillator input divided by 3 (4MHz input))
#pragma config IESO = OFF               // Internal External Switch Over Mode (IESO mode (Two-speed start-up)disabled)

// CONFIG1
#pragma config WDTPS = PS32768          // Watchdog Timer Postscaler (1:32,768)
#pragma config FWPSA = PR128            // WDT Prescaler (Prescaler ratio of 1:128)
#pragma config WINDIS = OFF             // Watchdog Timer Window (Standard Watchdog Timer enabled,(Windowed-mode is disabled))
#pragma config FWDTEN = OFF             // Watchdog Timer Enable (Watchdog Timer is disabled)
#pragma config ICS = PGx2               // Comm Channel Select (Emulator functions are shared with PGEC2/PGED2)
#pragma config GWRP = OFF               // General Code Segment Write Protect (Writes to program memory are allowed)
#pragma config GCP = OFF                // General Code Segment Code Protect (Code protection is disabled)
#pragma config JTAGEN = OFF             // JTAG Port Enable (JTAG port is disabled)

#include "umogi2.h"

char lcd[lcd_cpl*2+1];
BYTE uart_b[UART_BLEN],wp,rp;
int bf_max;

// irq
void __attribute__((__interrupt__, __auto_psv__)) _U1RXInterrupt (void)
{
    uart_b[wp++]=U1RXREG; if (wp==UART_BLEN) wp=0; // ciklikus buffer
    IFS0bits.U1RXIF=0; // clear if
}

void configI2C2(int f) 
{ // f khz
    int brg = (FCY/(f*1000L) - FCY/10000000L) - 1; 
    I2C2BRG = brg;          //I2C baudrate erteke 
    I2C2CONbits.I2CEN = 1;  //I2C endedelyezese 
}

void startI2C2(void) 
{ 
  I2C2CONbits.SEN = 1;      //start kuldese 
  while (I2C2CONbits.SEN);  //start befejezodott 
}

void rstartI2C2(void) 
{ 
  I2C2CONbits.RSEN = 1;     //restart kuldese 
  while (I2C2CONbits.RSEN); //restart befejezodott 
}

void stopI2C2(void) 
{ 
    I2C2CONbits.PEN=1;        //stop kuldese 
    while (I2C2CONbits.PEN);  //stop befejezodott 
}

BYTE putI2C2(BYTE val) // out: ACK?
{ 
    I2C2TRN = val;                // bajt kuldese 
    while (I2C2STATbits.TRSTAT);  // (8bit+ack) befejezodott 
    return I2C2STATbits.ACKSTAT; // 1: nak, 0: ack
}
BYTE getI2C2(BYTE ack) 
{ 
    BYTE in; 
    while (I2C2CON & 0x1F);         //varakozas idle allapotra 
    I2C2CONbits.RCEN = 1;           //fogadas engedelyezese 
    while (!I2C2STATbits.RBF);      //varakozas az adat erkezesere 
    in = I2C2RCV;               //1 bajt beolvasasa 
    //varakozas idle allapotra, nyugtazas kuldese elott 
    while (I2C2CON & 0x1F);         
    I2C2CONbits.ACKDT = ack;        //ACK bit beallitasa 
    I2C2CONbits.ACKEN = 1;          //ACK bit visszakuldese 
    while (I2C2CONbits.ACKEN);      //varakozas a befejezesig 
    return in;                 
}

void write_byte_I2C2(BYTE addr,BYTE data) 
{
    startI2C2();
    putI2C2(I2C_W(addr)); 
    putI2C2(data); 
    stopI2C2(); 
}

BYTE read_byte_I2C2(BYTE addr) 
{ 
    BYTE out;
    startI2C2(); 
    putI2C2(I2C_R(addr)); 
    out = getI2C2(I2C_NAK); //az utolso nyugtazas mindig NAK 
    stopI2C2();
    return out;
}

// output: ACK ok, NAK nem válaszolt
BYTE read_array_I2C2(BYTE addr,BYTE* t, BYTE darab)
{
    BYTE i,b;
    startI2C2(); 
    b=putI2C2(I2C_R(addr));
    if (b==I2C_ACK) // valaszolt i2c-n
      for (i=0; i<darab; i++)
      {
        if (i!=darab-1) t[i] = getI2C2(I2C_ACK); // ACK: kerem a kovetkezot
        else t[i]=getI2C2(I2C_NAK); // NAK: eleg volt
      }
    stopI2C2();
    return b;
}
float Read_Temp_LM75(void) 
{ 
    BYTE t[2]; 
    int temp; 
    startI2C2();
    putI2C2(I2C_W(lm75_addr)); // device addr, write
    putI2C2(0); // temp reg
    rstartI2C2();
    putI2C2(I2C_R(lm75_addr)); // device read
    t[0]=getI2C2(I2C_ACK); // msb, kov
    t[1]=getI2C2(I2C_NAK); // lsb, eleg.
    stopI2C2();
    temp = t[0] << 8;  // msb
    temp = (temp + t[1]) >> 5;  //lsb, 11bit, aritm. eltolas
    // D10 = 1 eseten negativ
    return temp*0.125; 
}

// output: 0:ack, 1 not ack
BYTE EEByteWrite(BYTE i2c_addr,BYTE addr, BYTE data)
{
   BYTE ack;
   startI2C2(); 
   ack=putI2C2(I2C_W(i2c_addr)); // i2c
   if (ack!=I2C_ACK)
   {
       stopI2C2();
       return ack;
   }
   putI2C2(addr); // eeprom cim
   putI2C2(data);
   stopI2C2();
   return I2C_ACK; //0
}

BYTE EERandomRead(BYTE i2c_addr,BYTE addr)
{
   BYTE ack,i;
   startI2C2(); 
   ack=putI2C2(I2C_W(i2c_addr)); // i2c
   if (ack!=I2C_ACK)
   {
       stopI2C2();
       return 0;
   }
   putI2C2(addr); // eeprom cim
   rstartI2C2();
   putI2C2(I2C_R(i2c_addr)); // i2c 
   i=getI2C2(I2C_NAK); 
   stopI2C2();
   return i;
}

BYTE EESequentialRead(BYTE i2c_addr,BYTE addr,BYTE *rdptr,BYTE length )
{
   BYTE i,ack;
   startI2C2(); 
   ack=putI2C2(I2C_W(i2c_addr)); // i2c
   if (ack!=I2C_ACK)
   {
       stopI2C2();
       return ack;
   }
   putI2C2(addr); // eeprom cim
   rstartI2C2();
   putI2C2(I2C_R(i2c_addr)); // i2c 
   for (i=0; i<length; i++)
   {
       if (i!=length-1) rdptr[i]=getI2C2(I2C_ACK); 
       else rdptr[i]=getI2C2(I2C_NAK);
   }
   stopI2C2();
   return I2C_ACK;
}


BYTE bcd2bin(BYTE b)
{
    BYTE x;
    x=((b & 0xf0)>>4)*10+(b & 0x0f);
    return x;
}

BYTE char2bin(BYTE c) {
  if (c<='9') return c-'0';
  return (c-'A'+10);
}


int WriteSPI1(int i) 
{   
    SPI1BUF = i; // buffer írása küldésre 
    while(!SPI1STATbits.SPIRBF) Nop(); // várakozás az átvitel befejezéséig 
    return SPI1BUF;    // beérkezõ adat kiolvasása 
}

BYTE ReadSR_25xx256(void) 
{
   BYTE i; 
   CSEE = 0; // EEPROM kiválasztása 
   WriteSPI1(SEE_RDSR);  // státusz regiszter olvasása
   i = WriteSPI1(0); // küldés/fogadás 
   CSEE = 1; // EEPROM elengedése 
   return i; 
}

void WriteEnable_25xx256(void) 
{ 
   CSEE = 0; // EEPROM kiválasztása 
   WriteSPI1(SEE_WENA); // írás engedélyezése parancs
   CSEE=1; // EEPROM elengedése
   Nop();Nop();
   CSEE=0;
   WriteSPI1(SEE_WRSR); // write stat reg
   WriteSPI1(0x00); // wpen 0, wp0, wp1 0
   CSEE = 1; // ettöl resetelõdik az írási engedély
   while (ReadSR_25xx256()  & 0x01) Nop();
   CSEE = 0;  WriteSPI1(SEE_WENA);  CSEE=1;
}

void WriteDisable_25xx256(void) 
{ 
   CSEE = 0; // EEPROM kiválasztása 
   WriteSPI1(SEE_WENA); // írás engedélyezése parancs
   CSEE=1; // EEPROM elengedése
   Nop();Nop();
   CSEE=0;
   WriteSPI1(SEE_WRSR); // write stat reg
   WriteSPI1(0xff); // wpen 1, wp1, wp1 1
   CSEE = 1;
   while (ReadSR_25xx256()  & 0x01) Nop();
}

BYTE ReadEE_25xx256(int address) 
{ 
    BYTE val; 
    CSEE = 0; // EEPROM kiválasztása 
    WriteSPI1(SEE_READ); // olvasás parancs 
    WriteSPI1(address>>8); // a cím felsõ része (MSB) 
    WriteSPI1(address & 0x00ff); // a cím alsó része (LSB) 
    val = WriteSPI1(0);     // dummy érték küldése/érték beolvasása 
    CSEE = 1;               // EEPROM elengedése 
    return val;
}

void ReadEEBlock_25xx256(int address,BYTE *t,int darab)
{
    int i;
    CSEE=0;
    WriteSPI1(SEE_READ); // olvasás parancs 
    WriteSPI1(address>>8); // a cím felsõ része (MSB) 
    WriteSPI1(address & 0x00ff); // a cím alsó része (LSB) 
    for (i=0;i<darab;i++) t[i]=WriteSPI1(0);
    CSEE=1;
}

void WriteEE_25xx256(int address, BYTE data) 
{ 
    WriteEnable_25xx256(); //Írás engedélyezése
    CSEE = 0; // EEPROM kiválasztása 
    WriteSPI1(SEE_WRITE); // írás parancs 
    WriteSPI1(address>>8); // a cím felsõ része (MSB) 
    WriteSPI1(address & 0x00ff); // a cím alsó része (LSB) 
    WriteSPI1(data); // az adat küldése 
    CSEE = 1; // EEPROM elengedése
    while (ReadSR_25xx256()  & 0x01) Nop(); // írás folyamatának vége (WIP) 
    WriteDisable_25xx256();
}

// dac a mikobuson. 8 bites SPI átvitel
void write_mcp4821(int adat)
{
    BYTE i;
    i=(adat & 0xff00)>>8; // H
    CS2=0; // L
    WriteSPI1(i);
    i=adat &0x00ff; // L
    WriteSPI1(i);
    CS2=1; // H
}

void lcd_data(BYTE b,BYTE rs)
{ // rs: 0:inst, 1:data 
   int trise_ment=TRISE, bf_lep=0;
   LCD_RS=rs;
   LCD_DATA=(LCD_DATA & 0xff00) | b;
   Nop(); Nop(); Nop(); // >230 ns. 62.5 ns/utasitĂĄs
   LCD_E=1;// +1 utasitĂĄs
   Nop(); Nop(); Nop();
   LCD_E=0;
   // bf check
   Nop(); Nop(); Nop();
   LCD_DATA &= 0xff00;
   TRISE |= 0x00ff; // lcd input
   LCD_RW=1; // r/w is 1
   LCD_RS=0;
   Nop(); Nop(); Nop();
   LCD_E=1;
   do
   {
       Nop();Nop();Nop();
       bf_lep++;
   } while (LCD_BF && bf_lep<0x1000);
   if (bf_lep>bf_max) bf_max=bf_lep;
   LCD_E=0;
   Nop();
   if (bf_lep>=0x1000) // 1729 
   {
       LEDR=1;
       __delay_ms(50);
       LEDR=0;
   }
   TRISE=trise_ment; 
   LCD_RW=0;
}

void lcd_init()
{
  // hitachi HD44780
  // tris
  LATE=0x00ff;
  LCD_E=0;
  bf_max=0;  
  TRISE &= 0xff00; // upper: don't change, lower 8: data output
  TRISGbits.TRISG0=0; // rs
  TRISGbits.TRISG1=0; // rw
  TRISFbits.TRISF1=0; // e
  // oled: OC
//  if (id==01) // umogi2-01, OLED
//  {
//    ODCGbits.ODG0=1;
//    ODCGbits.ODG1=1;
 //   ODCFbits.ODF1=1;
 //   ODCE|=0x00ff;
//    LED1=1;
 // }  
  LCD_RW=0;
  __delay_ms(50); // more than 40 msec
  // 8 bit: bf can be checked
  lcd_data(0b00111000,0); // 0x38 N=1 (2 lines), 5x8 font, charset 0 - ft1:0=00 
  lcd_data(0x0e,0); // 0b00001110: disp on, curs on, blink off 
  lcd_data(0x01,0); // disp clear
  lcd_data(0x02,0); // return home
  lcd_data(0x06,0); // entry mode set 
  lcd_cls();
}

void lcd_update()
{
    int i;
    lcd_data(0x80,0); // line 0
    for (i=0; i<lcd_cpl; i++)
        lcd_data(lcd[i],1); // lcd write
    lcd_data(0xc0,0); // line 1
    for (i=0; i<lcd_cpl; i++)
        lcd_data(lcd[lcd_cpl+i],1);
}

void lcd_cls()
{
    memset(lcd,' ',2*lcd_cpl);   
}

void lcd_cgram(void) 
{
  lcd_data(0x40,0); // set cgram, and define char 0
  lcd_data(0,1); // 0 8x5
  lcd_data(0x0a,1); // 1
  lcd_data(0x15,1); // 2 
  lcd_data(0x11,1); // 3 
  lcd_data(0x0a,1); // 4 
  lcd_data(0x04,1); // 5 
  lcd_data(0,1); // 6 
  lcd_data(0,1); // 7 
  // char 1 ...
}

BYTE uart_nextchar(void)
{
    BYTE c;
    while (wp==rp) Nop(); 
    c=uart_b[rp++]; if (rp==UART_BLEN) rp=0;
    return c;
}

void uart_send_byte(BYTE b)
{ 
    while ( U1STAbits.UTXBF) Nop();   //csak ha TX buffer üres 
    U1TXREG = b; 
}

void uart_init(int baud)
{
    long brg=(long)(FCY)/(4*(long)baud)-1;
    // uart init
    U1BRG= brg & 0xffff; 
    wp=0; rp=0; // a ciklikus buffer pointerei
    U1MODE=0xab08; // 0b1010 1011 0000 1000;//on-no irda-rxtx-standard mode, 8n1, brgh=1
    U1STAbits.UTXEN=1;// enable tx
    
    IFS0bits.U1RXIF=0;
    IPC2bits.U1RXIP=1;
    IEC0bits.U1RXIE=1;
}

void uart_printf(const char *s)
{
    BYTE b;
    while ((b=*s))
    {
        uart_send_byte(b);
        s++;
    }
}